<#
    .SYNOPSIS
    PRTG PowerShell script to monitor individual Proxmox VMs and Containers

    .DESCRIPTION
    Monitors CPU, Memory, and Status for each VM and Container in Proxmox VE

    .NOTES
    Version 1.4 - Corrected parameter splitting logic
    
    Requires:
    - PowerShell 7: https://aka.ms/powershell-release?tag=lts
    - Corsinvest.ProxmoxVE.Api module: Install-Module Corsinvest.ProxmoxVE.Api

    Author: Created for PRTG monitoring
#>
param(
    [string] $Server = "",
    [string] $Port = "8006",
    [string] $APITOKEN = '',
    [switch] $SkipCertCheck,
    [string] $ExcludeName = '',
    [string] $IncludeName = '',
    [string] $ExcludeVMID = '',
    [string] $IncludeVMID = '',
    [string] $ExcludeType = '',
    [string] $IncludeType = '',
    [string] $AcknowledgedStopped = ''
)

# Catch all unhandled errors
trap {
    $Output = "line:$($_.InvocationInfo.ScriptLineNumber.ToString()) char:$($_.InvocationInfo.OffsetInLine.ToString()) --- message: $($_.Exception.Message.ToString()) --- line: $($_.InvocationInfo.Line.ToString()) "
    $Output = $Output.Replace("<", "")
    $Output = $Output.Replace(">", "")
    Write-Output "<prtg>"
    Write-Output "<error>1</error>"
    Write-Output "<text>$Output</text>"
    Write-Output "</prtg>"
    Exit
}

$ErrorActionPreference = "Stop"

# Check if PowerShell 7 is installed
if (-not (Test-Path "C:\Program Files\PowerShell\7\pwsh.exe")) {
    Write-Output "<prtg>"
    Write-Output "<error>1</error>"
    Write-Output "<text>Please verify PowerShell 7 is installed under C:\Program Files\PowerShell\7\pwsh.exe</text>"
    Write-Output "</prtg>"
    Exit
}

# PowerShell 7 workaround for PRTG (runs PS5 by default)
if ($PSVersionTable.PSVersion.Major -eq 5) {
    if ($myInvocation.Line) {
        [string]$output = &'C:\Program Files\PowerShell\7\pwsh.exe' -NonInteractive -NoProfile -CommandWithArgs "$($myInvocation.Line)"
    }
    else {
        [string]$output = &'C:\Program Files\PowerShell\7\pwsh.exe' -NonInteractive -NoProfile -file "$($myInvocation.InvocationName)" $args
    }
    Write-Output $output
    exit
}

# Import module
try {
    Import-Module "Corsinvest.ProxmoxVE.Api" -ErrorAction Stop
}
catch {
    Write-Output "<prtg>"
    Write-Output "<error>1</error>"
    Write-Output "<text>Error loading Corsinvest.ProxmoxVE.Api module: $($_.Exception.Message)</text>"
    Write-Output "</prtg>"
    Exit
}

# Check credentials
if (($APITOKEN -eq "") -or ($null -eq $APITOKEN)) {
    Write-Output "<prtg>"
    Write-Output "<error>1</error>"
    Write-Output "<text>APITOKEN missing</text>"
    Write-Output "</prtg>"
    Exit
}

# Check server
if (($Server -eq "") -or ($null -eq $Server)) {
    Write-Output "<prtg>"
    Write-Output "<error>1</error>"
    Write-Output "<text>Server not specified</text>"
    Write-Output "</prtg>"
    Exit
}

# Connect to server
try {
    if ($SkipCertCheck) {
        $ticket = Connect-PveCluster -ApiToken $APITOKEN -HostsAndPorts "$($Server):$($Port)" -SkipCertificateCheck
    }
    else {
        $ticket = Connect-PveCluster -ApiToken $APITOKEN -HostsAndPorts "$($Server):$($Port)"
    }
}
catch {
    Write-Output "<prtg>"
    Write-Output "<error>1</error>"
    Write-Output "<text>Could not connect to PVE server $Server. Error: $($_.Exception.Message)</text>"
    Write-Output "</prtg>"
    Exit
}

# Verify connection
if (($null -eq $ticket) -or ($ticket.ApiToken -eq "")) {
    Write-Output "<prtg>"
    Write-Output "<error>1</error>"
    Write-Output "<text>Could not connect to PVE server $Server. No valid ticket received.</text>"
    Write-Output "</prtg>"
    Exit
}

# Get all VMs and containers
$all_vms = Get-PveVm -PveTicket $ticket
$all_vms = $all_vms | Where-Object { $_.template -eq "0" }

# Apply filters
if ($ExcludeName -ne "") {
    $all_vms = $all_vms | Where-Object { $_.Name -notmatch $ExcludeName }
}
if ($IncludeName -ne "") {
    $all_vms = $all_vms | Where-Object { $_.Name -match $IncludeName }
}
if ($ExcludeVMID -ne "") {
    $all_vms = $all_vms | Where-Object { $_.vmid -notmatch $ExcludeVMID }
}
if ($IncludeVMID -ne "") {
    $all_vms = $all_vms | Where-Object { $_.vmid -match $IncludeVMID }
}
if ($ExcludeType -ne "") {
    $all_vms = $all_vms | Where-Object { $_.type -notmatch $ExcludeType }
}
if ($IncludeType -ne "") {
    $all_vms = $all_vms | Where-Object { $_.type -match $IncludeType }
}

# Parse acknowledged stopped list
$ackStoppedList = @()
if ($AcknowledgedStopped -ne '') {
    # Split by space, as this is how PRTG passes the parameter list
    $ackStoppedList = $AcknowledgedStopped -split ' ' | ForEach-Object { $_.Trim() }
}

# Start XML output
$xmlOutput = '<prtg>'

# Counters
$runningCount = 0
$stoppedCount = 0
$unknownCount = 0

foreach ($vm in $all_vms) {
    $vmName = $vm.Name
    $vmId = $vm.vmid
    $vmType = $vm.type
    $vmNode = $vm.node
    $vmStatus = $vm.status
    
    # Type label
    $typeLabel = if ($vmType -eq "qemu") { "VM" } else { "CT" }
    
    # Determine status: 2 = running, 3 = stopped (ack), 1 = stopped, 0 = unknown
    if ($vmStatus -eq "running") {
        $statusValue = 2
        $runningCount++
    }
    elseif ($vmStatus -eq "stopped") {
        if ($vmId.ToString() -in $ackStoppedList) {
            $statusValue = 3  # Stopped (Acknowledged)
        }
        else {
            $statusValue = 1  # Stopped
        }
        $stoppedCount++
    }
    else {
        $statusValue = 0
        $unknownCount++
    }
    
    # Status channel
    $xmlOutput += "
<result>
<channel>$typeLabel $vmName ($vmId) - Status</channel>
<value>$statusValue</value>
<unit>Custom</unit>
<customunit></customunit>
<ValueLookup>prtg.custom.proxmox.vmstatus</ValueLookup>
</result>"
    
    # Only get CPU/Memory for running VMs/containers
    if ($vmStatus -eq "running") {
        try {
            if ($vmType -eq "qemu") {
                $vmDetails = (Get-PveNodesQemuStatusCurrent -PveTicket $ticket -Node $vmNode -VmId $vmId).Response.data
            }
            else {
                $vmDetails = (Get-PveNodesLxcStatusCurrent -PveTicket $ticket -Node $vmNode -VmId $vmId).Response.data
            }
            
            # CPU Usage
            if ($null -ne $vmDetails.cpu) {
                $cpuPercent = [math]::Round($vmDetails.cpu * 100, 2)
                $xmlOutput += "
<result>
<channel>$typeLabel $vmName ($vmId) - CPU</channel>
<value>$cpuPercent</value>
<unit>Percent</unit>
<float>1</float>
</result>"
            }
            
            # Memory Usage
            if (($null -ne $vmDetails.mem) -and ($null -ne $vmDetails.maxmem) -and ($vmDetails.maxmem -gt 0)) {
                $memPercent = [math]::Round(($vmDetails.mem / $vmDetails.maxmem) * 100, 2)
                $xmlOutput += "
<result>
<channel>$typeLabel $vmName ($vmId) - Memory</channel>
<value>$memPercent</value>
<unit>Percent</unit>
<float>1</float>
</result>"
            }
        }
        catch {
            # Skip if we can't get details - VM might have just stopped
        }
    }
}

# Summary channels
$xmlOutput += "
<result>
<channel>Total Running</channel>
<value>$runningCount</value>
<unit>Count</unit>
</result>
<result>
<channel>Total Stopped</channel>
<value>$stoppedCount</value>
<unit>Count</unit>
</result>
<result>
<channel>Total Unknown</channel>
<value>$unknownCount</value>
<unit>Count</unit>
</result>"

$xmlOutput += "<text>Monitoring $($all_vms.Count) VMs/Containers</text>"
$xmlOutput += "</prtg>"

Write-Output $xmlOutput
